/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/

#undef TRACE_ENABLE		/* Enable/disable debugging in this module */

#include "lib.h"
#include "uilib.h"
#include "nvram.h"
#include "platform.h"		/* For prompt string */

/*--------------------------------------------------------------------*/
/* simple routine to display in the manner of dmesg the diagnostic record */

#define LINESPAGE 20
#define LINEBUFLEN 76			/* A tidy line is no longer than this */

static const Point p_inscrn = { 3, 2 };

/* Copy to a buffer all input up to a null character or newline */
/* Stop if the max number of bytes is transferred */
/* Return the number of bytes consumed (including newlines, which are cut) */

static int getnstr( const char *src, char *dest, const size_t max )
{
   int i,j;

   for ( i=0, j=0; i < max - 1; i++, j++ )
   {
	/* Strip out any newlines or carriage returns */
	if ( src[i] == '\r' || src[i] == '\n' )
	{
	    i++;

	    /* handle a CR-LF pairing */
	    if ( (i < max - 1) && (src[i] == '\r' || src[i] == '\n') )
		i++;

	    break;
	}

	/* have we reached the end of our source string? */
	if ( src[i] == '\0' )
	    break;

	/* Copy across our legitimate character */
	dest[ j ] = src[ i ];
    }

    /* terminate the string and get back */
    dest[j] = '\0';
    return i;
}



/* display a particular log record selected by the user */

static void display_log_record( nvlog_t *rec )
{
    uint8 *ptr;
    int lines;
    size_t bytes, maxb, remain;
    int key;
    char linebuf[ LINEBUFLEN ];


    /* Skip past ASCII-magic log header */
    bytes = strlen( NVMAGIC );
    ptr = rec->rec;

    do {

	mobo_box( r_scrn, "Non-volatile Diagnostics Record" );
	mobo_goto( p_inscrn );

	/* Display a screenful, couting the lines */
	for ( lines=0; lines < LINESPAGE; lines++ )
	{
	    remain = rec->len - bytes;
	    maxb = LINEBUFLEN > remain ? remain : LINEBUFLEN;
	    bytes += getnstr( rec->rec + bytes, linebuf, maxb );
	    printf_dbm( "%s\r", linebuf );
	}

	printf_dbm( "\rPress any key to continue (or 'q' to quit) [%d/%d]",
		bytes, rec->len );
	key = mobo_key( 0 );
	if ( tolower( key ) == 'q' )
	    return;

    } while ( bytes < rec->len-1);
}

DBM_STATUS logread( void )
{
    nvlog_t recs[ LINESPAGE ], *last=NULL;
    unsigned nrecs;
    int i=0;
    DBM_STATUS sval;
    int choice, rval;
    char line[ LINEBUFLEN ];
    BOOLEAN more_records = TRUE;
    size_t magiclen = strlen( NVMAGIC );


    /* check that the logstream isn't going to the ROM as this snarls things */
    if ( nvlog_state != NVLOG_CLOSED )
    {
	mobo_alertf( "Can't do that right now",
	    "Sorry, the NVRAM log can't be read because it is still in use\r"
	    "(log state is %d, closed state is %d)",
	    nvlog_state, NVLOG_CLOSED );
	return STATUS_FAILURE;
    }
    

    sval = nvlog_open( NVLOG_READ );
    if ( sval == STATUS_FAILURE )
    {
	mobo_alertf( "No Non-volatile Log Support",
		"This platform either does not support for NVRAM log records\r"
		"Or it is not functioning correctly" );
	return STATUS_FAILURE;
    }


    do {

	/* Obtain a fresh batch of records (if available) */
	for ( i=0; i < LINESPAGE; i++ )
	{
	    nvlog_next( last, recs + i );
	    if ( recs[i].rec == NULL )		/* End of records */
	    {
		more_records = FALSE;
		break;
	    }
	    last = recs + i;
	}
	nrecs = i;

	/* Present this screen's worth for the user to interact with */
	do {

	    /* Display a screen's worth of records (if available) */
	    mobo_box( r_scrn, "Non-volatile Diagnostics Record" );
	    mobo_goto( p_inscrn );

	    for ( i=0; i < nrecs; i++ )
	    {
		getnstr( recs[i].rec + magiclen, line, LINEBUFLEN );
		printf_dbm( "%2d) %s\r", i, line );
	    }
	    if ( nrecs == 0 )
		printf_dbm( "No log records were found in the ROM" );

	    /* Prompt the User for options */
	    mobo_goto( p_help );
	    printf_dbm( "Enter session number "
			"[Return for next page, 'quit' to cancel]:" );
	    mobo_goto(p_prompt);
	    printf_dbm( Prompt );		/* platform prompt string */
	    rval = mobo_input( line, sizeof( line ) );

	    /* Act accordingly... */
	    /* 0 chars entered signifies return signifies next screenful */
	    if ( rval == 0 )
		break;

	    /* "quit" selected? */
	    if ( strcmp( line, "quit" ) == 0 )
	    {
		nvlog_close();
		return STATUS_SUCCESS;
	    }

	    /* Valid option chosen? */
	    choice = -1;
	    sscanf( line, "%d", &choice );
	    if ( choice >= 0 && choice < nrecs )
		display_log_record( recs + choice );

	} while ( TRUE );

    } while ( more_records );

    nvlog_close();
    return STATUS_SUCCESS;
}
