/**********************************************************/
/* IBM 3278 keyboard to PS/2                              */
/*                                                        */
/* Copyright (c) 2001, Jamie Honan                        */
/*                                                        */
/* All rights reserved.                                   */
/*                                                        */
/* This program is free software; you can redistribute it */
/* and/or modify it under the terms of the GNU General    */
/* Public License as published by the Free Software       */
/* Foundation; either version 2 of the License, or        */
/* (at your option) any later version.                    */
/*                                                        */
/* This program is distributed in the hope that it will   */
/* be useful, but WITHOUT ANY WARRANTY; without even the  */
/* implied warranty of MERCHANTABILITY or FITNESS FOR A   */
/* PARTICULAR PURPOSE.  See the GNU General Public        */
/* License for more details.                              */
/*                                                        */
/* You should have received a copy of the GNU General     */
/* Public License along with this program; if not, write  */
/* to the Free Software Foundation, Inc.,                 */
/* 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA */
/*                                                        */
/* Licence can be viewed at                               */
/* http://www.fsf.org/licenses/gpl.txt                    */
/*                                                        */
/**********************************************************/

#include <avr/io.h>
#include <avr/interrupt.h>

#include "global.h"
#include "IBM_keyboard.h"
#include "keysta.h"

/* This is the ISR which handles the keyboard communication
   between the keyboard emulator (DEVICE) and the PC (HOST). 
   The interface routines and flag bits are commented in keysta.h

   This routine reads and drives the clock and data signals.

   Note that care should be take with other routines that
   set DDR registers, or pin out registers that share the ports
   that are used here.

   In certain circumstances this read - modify - write cycle 
   should be protected by disabling interrupts.
 */

#define COUNT_UP (256 - ((33 * CYCLES_PER_US)/8))

static void clockHigh(void) {         
   sbi(PORTB, PS2_CLK_OUT);           // set PS/2 Clock high.
}

static void clockLow(void) {
   cbi(PORTB, PS2_CLK_OUT);           // set PS/2 Clock low.
}

static void dataHigh(void) {
   sbi(PORTD, PS2_DATA_OUT);          // set PS/2 Data high.
}

static void dataLow(void) {
   cbi(PORTD, PS2_DATA_OUT);          // set PS/2 Data low.
}

#define readClockPin() (PIND & (1 << PS2_CLK_INP))
#define readDataPin()  (PIND & (1 << PS2_DATA_INP))

volatile unsigned char kbd_flags;
   /* FLA_CLK_HIGH   0x01
      FLA_RX_BAD     0x02
      RX_BUF_FULL    0x04
      TX_BUF_EMPTY   0x08
      FLA_TX_OK      0x10 */

enum enum_state { 
   // Idle states
   IDLE_START = 0, 
   IDLE_WAIT_REL, 
   IDLE_OK_TO_TX, 
   IDLE_END,
   // Receive states  
   RX_START = IDLE_END+10, 
   RX_RELCLK, 
   RX_DATA0, RX_DATA1, RX_DATA2, RX_DATA3, 
   RX_DATA4, RX_DATA5, RX_DATA6, RX_DATA7,
   RX_PARITY, 
   RX_STOP, 
   RX_SENT_ACK, 
   RX_END,
   // Transmit states 
   TX_START = RX_END+10, 
   TX_DATA1, TX_DATA2, TX_DATA3, 
   TX_DATA4, TX_DATA5, TX_DATA6, TX_DATA7,
   TX_PARITY, 
   TX_STOP, 
   TX_AFTER_STOP, 
   TX_END 
};

volatile unsigned char kbd_state;

static volatile unsigned char   rx_byte;
static volatile unsigned char   tx_byte;
static volatile unsigned char   tx_shift;
static volatile unsigned char   parity;

void init_kbd(void) {
   kbd_state = IDLE_START;
   kbd_flags = FLA_CLK_HIGH | FLA_TX_OK | TX_BUF_EMPTY;

   clockHigh();
   dataHigh();

   TCCR0A = _BV(WGM01);   // Mode = CTC
   TCCR0B = _BV(CS01);    // F_CPU / 8 = 1,843MHz ==> 0,5425 us
   OCR0A  =  74;          // 0,5425 * 77 = 42 us between ticks ==> 12 kHz SIG_OUTPUT_COMPARE0A
   TIMSK0 = _BV(OCIE0A);  // Enable Interrupt TimerCounter0 Compare Match A (SIG_OUTPUT_COMPARE0A)

}

// Transmit one byte to the HOST.
void kbd_set_tx(unsigned char txchar) {
   cli(); 
   tx_byte = txchar;
   kbd_flags &= ~FLA_TX_OK;    // Reset TX OK flag 
   kbd_flags &= ~TX_BUF_EMPTY; // Clear buffer empty flag 
   sei();
}

// Receive one byte from the HOST.
unsigned char kbd_get_rx_char(void) {
   cli();
   kbd_flags &= ~RX_BUF_FULL;  // Reset buffer full flag 
   sei();
   return rx_byte;
}


// *********************************************
// ***   Timer 0 interrupt handler routine.  ***
// *********************************************

ISR(SIG_OUTPUT_COMPARE0A) {
  // Restart the timer...
  // outp(COUNT_UP, TCNT2);     // Value counts up from this to zero */

   if (kbd_state < IDLE_END) {  // Start, wait_rel or ready to tx

      // ****************************
      // *** Process IDLE states. ***
      // ****************************
      dataHigh();
      if (!(kbd_flags & FLA_CLK_HIGH)) {
         kbd_flags |= FLA_CLK_HIGH;
         clockHigh();
         return;
      }
      
      // If clock is held low, then we must prepare to receive
      if (!readClockPin()) {
         kbd_state = IDLE_WAIT_REL;
         return;
      }
      
      switch(kbd_state) {
         case IDLE_START:
            kbd_state = IDLE_OK_TO_TX;
            return;
         case IDLE_WAIT_REL:
            if (!readDataPin()) {
               // HOST wants to transmit
               kbd_state = RX_START;
               return;
            }
            // Just an ack or something
            kbd_state = IDLE_OK_TO_TX;
            return;
         case IDLE_OK_TO_TX:
            if (!(kbd_flags & TX_BUF_EMPTY) ) {
               // Start transmitting... 
               dataLow();
               kbd_state = TX_START;
            }
            return;
      }
      return;

   } else { // end < IDLE_END
      
      if (kbd_state < RX_END) {

         // *******************************
         // *** Process RECEIVE states. ***
         // *******************************
         if (!(kbd_flags & FLA_CLK_HIGH)) {
            kbd_flags |= FLA_CLK_HIGH;
            clockHigh();
            return;
         }
         // At this point clock is high in preparation to going low
         if (!readClockPin()) {
            // HOST is still holding clock down
            dataHigh();
            kbd_state = IDLE_WAIT_REL;
            return;
         }

         switch(kbd_state) {
            case RX_START:
               // HOST has released clock line
               // DEVICE keeps it high for a half cycle
               kbd_flags &= ~FLA_RX_BAD;
               kbd_state++;   // Jump to next state            
               return;
            case RX_RELCLK:
               // Now PC has seen clock high, show it some low
               break;
            case RX_DATA0:
//               kbd_flags &= ~FLA_RX_BYTE;
               if (readDataPin()) {
                  parity = 1;
                  rx_byte = 0x80; 
               } else {
                  parity = 0;
                  rx_byte = 0x00;
               }
               break;    // End clk hi 1 
            case RX_DATA1: 
            case RX_DATA2: 
            case RX_DATA3: 
            case RX_DATA4: 
            case RX_DATA5: 
            case RX_DATA6: 
            case RX_DATA7: 
               rx_byte >>= 1;
               if (readDataPin()) {
                  rx_byte |= 0x80;
                  parity++;
               }
               break;    // End clk hi 2 to 8
            case RX_PARITY: 
               if (readDataPin()) {
                  parity++;
               }
               if (!(parity & 0x01)) {
                  // Error, not odd parity
                  kbd_flags |= FLA_RX_BAD;
               }
               break;    // End clk hi 9
            case RX_STOP: 
               if (!readDataPin()) {
                  // Missing stop bit
                  kbd_flags |= FLA_RX_BAD;
               }
               if (!(kbd_flags & FLA_RX_BAD)) {
                  dataLow();
                  kbd_flags |= RX_BUF_FULL;
               }
               break;    // End clk hi 10 */
            case RX_SENT_ACK: 
               dataHigh();
               kbd_state = IDLE_START;
               return;   // Remains in clk hi 11 */
         }
         
         clockLow();
         kbd_flags &= ~(FLA_CLK_HIGH);
         kbd_state++;    // Jump to next state            
         return;

      } else { // end < RX_END

         // ********************************
         // *** Process TRANSMIT states. ***
         // ********************************
         if (kbd_state < TX_END) {
            if (kbd_flags & FLA_CLK_HIGH) {
               if (!readClockPin()) {
                  // HOST is still holding down clock
                  dataHigh();
                  kbd_state = IDLE_WAIT_REL;
                  return;
               }
               kbd_flags &= ~FLA_CLK_HIGH;
               clockLow();
               return;
            }

            // At this point clock is low in preparation to go high
            kbd_flags |= FLA_CLK_HIGH;
            clockHigh();
            switch(kbd_state) {
               case TX_START:
                  tx_shift = tx_byte;
                  parity = 0;
               case TX_DATA1: 
               case TX_DATA2: 
               case TX_DATA3: 
               case TX_DATA4:
               case TX_DATA5: 
               case TX_DATA6: 
               case TX_DATA7:
                  if (tx_shift & 0x01) {
                     dataHigh();
                     parity++;
                  } else {
                     dataLow();
                  }
                  tx_shift = tx_shift >> 1;  // Next data bit
                  break;   // Clock hi 1 to 8 */
               case TX_PARITY: 
                  if (parity & 0x01) {
                     dataLow();
                  } else {
                     dataHigh();
                  }
                  kbd_flags |= FLA_TX_OK;
                  break;   // Clock hi 9 */
               case TX_STOP: 
                  dataHigh();
                  break;   // Clock hi 10 */
               case TX_AFTER_STOP:
                  kbd_flags |= TX_BUF_EMPTY; // Transmission completed. 
                  kbd_state = IDLE_START;
                  return;  // Remains in clk hi 11 */ 
            }

            kbd_state++;   // Jump to next state            

         } // end of else
      }
   }
}
